use anyhow::{anyhow, Result};
use reqwest::Client;
use serde_json::{json, Value};

pub struct CedarClient {
    client: Client,
    base_url: String,
    app_id: String,
    app_secret: String,
    sdk_version: String,
    pub token: String,
    pub sdk_token: String,
    pub entitlement: String,
    pub signature_b64: String,
}

impl CedarClient {
    pub fn new(base_url: &str, app_id: &str, app_secret: &str, sdk_version: &str) -> Self {
        Self {
            client: Client::new(),
            base_url: base_url.trim_end_matches('/').to_string(),
            app_id: app_id.to_string(),
            app_secret: app_secret.to_string(),
            sdk_version: sdk_version.to_string(),
            token: String::new(),
            sdk_token: String::new(),
            entitlement: String::new(),
            signature_b64: String::new(),
        }
    }

    async fn request(&self, path: &str, method: reqwest::Method, body: Option<Value>, bearer: Option<&str>) -> Result<Value> {
        let mut req = self.client.request(method, format!("{}{}", self.base_url, path));
        if let Some(token) = bearer {
            req = req.bearer_auth(token);
        }
        if let Some(b) = body {
            req = req.json(&b);
        }
        let res = req.send().await?;
        let status = res.status();
        let payload: Value = res.json().await.unwrap_or_else(|_| json!({}));
        if !status.is_success() {
            return Err(anyhow!(payload.get("error").and_then(Value::as_str).unwrap_or("request failed")));
        }
        Ok(payload)
    }

    pub async fn health(&self) -> Result<Value> { self.request("/health", reqwest::Method::GET, None, None).await }

    pub async fn login(&mut self, username: &str, password: &str) -> Result<Value> {
        let out = self.request("/api/auth/login", reqwest::Method::POST, Some(json!({"username": username, "password": password})), None).await?;
        self.token = out.get("access_token").and_then(Value::as_str).unwrap_or_default().to_string();
        Ok(out)
    }

    pub async fn heartbeat(&self) -> Result<Value> {
        self.request("/api/auth/heartbeat", reqwest::Method::POST, Some(json!({})), Some(&self.token)).await
    }

    pub async fn sdk_init(&mut self, device_id: &str, hwid: &str) -> Result<Value> {
        let out = self.request("/api/sdk/init", reqwest::Method::POST, Some(json!({
            "app_id": self.app_id,
            "app_secret": self.app_secret,
            "sdk_version": self.sdk_version,
            "device_id": device_id,
            "hwid": hwid,
            "nonce": "rsdknonce1234567890abcd",
            "timestamp": chrono::Utc::now().timestamp()
        })), None).await?;
        self.sdk_token = out.get("token").and_then(Value::as_str).unwrap_or_default().to_string();
        Ok(out)
    }

    pub async fn activate_license(&mut self, license_key: &str, device_id: &str, hwid: &str) -> Result<Value> {
        let out = self.request("/api/sdk/license/activate", reqwest::Method::POST, Some(json!({
            "license_key": license_key,
            "device_id": device_id,
            "hwid": hwid,
            "nonce": "rsdkactnonce1234567890ab",
            "timestamp": chrono::Utc::now().timestamp()
        })), Some(&self.sdk_token)).await?;
        self.entitlement = out.get("entitlement").and_then(Value::as_str).unwrap_or_default().to_string();
        self.signature_b64 = out.get("signature_b64").and_then(Value::as_str).unwrap_or_default().to_string();
        Ok(out)
    }

    pub async fn sdk_heartbeat(&self) -> Result<Value> {
        self.request("/api/sdk/heartbeat", reqwest::Method::POST, Some(json!({})), Some(&self.sdk_token)).await
    }

    pub async fn validate(&self) -> Result<Value> {
        let q = format!("/api/sdk/license/validate?entitlement={}&signature_b64={}", urlencoding::encode(&self.entitlement), urlencoding::encode(&self.signature_b64));
        self.request(&q, reqwest::Method::GET, None, None).await
    }
}
