from __future__ import annotations

import base64
import hashlib
import hmac
import json
import os
import platform
import secrets
import time
import uuid
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Tuple

import requests

try:
    from nacl.signing import VerifyKey  # type: ignore

    _NACL_AVAILABLE = True
except Exception:
    _NACL_AVAILABLE = False


@dataclass
class UserData:
    username: str = ""
    role: str = ""
    hwid: str = ""
    ip: str = ""


@dataclass
class Subscription:
    plan: str = ""
    level: str = ""
    expiry: int = 0


@dataclass
class Response:
    success: bool = False
    message: str = ""
    http_code: int = 0
    raw: str = ""
    user_data: UserData = field(default_factory=UserData)
    subscriptions: List[Subscription] = field(default_factory=list)
    variables: Dict[str, str] = field(default_factory=dict)


@dataclass
class SecurityPolicy:
    require_https: bool = False
    allow_localhost_http: bool = True
    require_strong_app_secret: bool = True
    min_app_secret_len: int = 24
    max_clock_skew_secs: int = 180
    verify_entitlement_signature: bool = False
    require_pinned_server_signing_key: bool = False
    expected_server_signing_key_b64: str = ""
    block_if_debugger_detected: bool = False
    block_if_vm_detected: bool = False
    block_if_suspicious_env: bool = False
    enforce_license_prefix: bool = False
    accepted_license_prefixes: List[str] = field(default_factory=lambda: ["LIC-", "CEDAR-"])
    sign_heartbeat_requests: bool = True
    require_activation_validate_roundtrip: bool = True
    require_subscription_not_expired: bool = True
    min_session_ttl_secs: int = 120
    min_auth_ttl_secs: int = 120
    require_signed_custom_calls: bool = False


class CedarSDK:
    def __init__(
        self,
        app_name: str,
        owner_id: str,
        app_secret: str,
        version: str,
        base_url: str,
        verify_tls: bool = True,
    ) -> None:
        self.app_name = app_name
        self.owner_id = owner_id
        self.app_secret = app_secret
        self.version = version
        self.base_url = base_url.rstrip("/")
        if self.base_url.startswith("http://localhost"):
            self.base_url = self.base_url.replace("http://localhost", "http://127.0.0.1", 1)
        self.verify_tls = verify_tls

        self.response = Response()
        self.policy = SecurityPolicy()

        self.session_token = ""
        self.auth_token = ""
        self.session_expiry = 0
        self.auth_expiry = 0
        self.entitlement = ""
        self.signature_b64 = ""
        self.server_signing_key_b64 = ""
        self.encryption_mode = "ed25519_hmac"
        self.signature_alg = "hmac-sha256"
        self.last_license_key = ""
        self.variable_store: Dict[str, str] = {}
        self.default_headers: Dict[str, str] = {}
        self.owner_hint: str = ""
        self.discord_tag: str = ""

        self.endpoints: Dict[str, str] = {
            "sdk_init": "/api/sdk/init",
            "auth_login": "/api/auth/login",
            "auth_register": "/api/auth/register",
            "sdk_activate": "/api/sdk/license/activate",
            "sdk_deactivate": "/api/sdk/license/deactivate",
            "sdk_heartbeat": "/api/sdk/heartbeat",
            "sdk_validate": "/api/sdk/license/validate",
            "sdk_announcements": "/api/sdk/announcements",
            "sdk_motd": "/api/sdk/motd",
            "sdk_log": "/api/sdk/log",
            "auth_heartbeat": "/api/auth/heartbeat",
            "auth_logout": "/api/auth/logout",
            "public_license_abuse_report": "/api/public/license-abuse-report",
            "public_announcements": "/api/public/announcements",
            "public_motd": "/api/public/motd",
            "public_status": "/api/public/status",
            "public_runtime_config": "/api/public/runtime-config",
        }

    # ---------- Setup ----------
    def set_security_policy(self, policy: SecurityPolicy) -> None:
        self.policy = policy

    def set_signing_key_pin(self, expected_key_b64: str, require_pin: bool) -> None:
        self.policy.expected_server_signing_key_b64 = expected_key_b64
        self.policy.require_pinned_server_signing_key = require_pin

    def set_local_guards(self, block_debugger: bool, block_vm: bool) -> None:
        self.policy.block_if_debugger_detected = block_debugger
        self.policy.block_if_vm_detected = block_vm

    def set_endpoint(self, name: str, path: str) -> None:
        if name and path:
            self.endpoints[name] = path

    def set_default_header(self, name: str, value: str) -> None:
        if name:
            self.default_headers[name] = value

    def set_default_headers(self, headers: Dict[str, str]) -> None:
        for k, v in headers.items():
            if k:
                self.default_headers[k] = v

    def endpoint(self, name: str) -> str:
        return self.endpoints.get(name, "")

    def load_profile_from_env(self, prefix: str) -> bool:
        app_name = os.getenv(f"{prefix}_APP_NAME", "")
        owner_id = os.getenv(f"{prefix}_OWNER_ID", "")
        app_secret = os.getenv(f"{prefix}_APP_SECRET", "")
        version = os.getenv(f"{prefix}_VERSION", "")
        base_url = os.getenv(f"{prefix}_BASE_URL", "")
        if not all([app_name, owner_id, app_secret, version, base_url]):
            return False
        self.app_name = app_name
        self.owner_id = owner_id
        self.app_secret = app_secret
        self.version = version
        self.base_url = base_url.rstrip("/")
        if self.base_url.startswith("http://localhost"):
            self.base_url = self.base_url.replace("http://localhost", "http://127.0.0.1", 1)
        return True

    def save_profile(self, path: str) -> bool:
        try:
            body = {
                "app_name": self.app_name,
                "owner_id": self.owner_id,
                "app_secret": self.app_secret,
                "version": self.version,
                "base_url": self.base_url,
            }
            with open(path, "w", encoding="utf-8") as f:
                json.dump(body, f, ensure_ascii=True, indent=2)
            return True
        except Exception:
            return False

    def load_profile(self, path: str) -> bool:
        try:
            with open(path, "r", encoding="utf-8") as f:
                body = json.load(f)
            self.app_name = str(body.get("app_name", self.app_name))
            self.owner_id = str(body.get("owner_id", self.owner_id))
            self.app_secret = str(body.get("app_secret", self.app_secret))
            self.version = str(body.get("version", self.version))
            self.base_url = str(body.get("base_url", self.base_url)).rstrip("/")
            if self.base_url.startswith("http://localhost"):
                self.base_url = self.base_url.replace("http://localhost", "http://127.0.0.1", 1)
            return True
        except Exception:
            return False

    def set_owner_hint(self, owner_hint: str) -> None:
        self.owner_hint = owner_hint.strip()

    def set_discord_tag(self, discord_tag: str) -> None:
        self.discord_tag = discord_tag.strip()

    def detect_discord_tag(self) -> str:
        # Discord no longer guarantees globally visible tags from local machine data.
        # Use explicit env values first, then return empty when unavailable.
        for key in ("CEDAR_DISCORD_TAG", "DISCORD_TAG", "DISCORD_USERNAME"):
            val = os.getenv(key, "").strip()
            if val:
                self.discord_tag = val
                return val
        return ""

    def resolved_owner_hint(self) -> str:
        if self.owner_hint:
            return self.owner_hint
        if self.discord_tag:
            return self.discord_tag
        detected = self.detect_discord_tag()
        if detected:
            return detected
        local_user = os.getenv("USER", "").strip() or os.getenv("USERNAME", "").strip()
        if local_user:
            return local_user
        xff = self.default_headers.get("X-Forwarded-For", "").strip()
        if xff:
            return f"ip:{xff}"
        return ""

    # ---------- Security/Utility ----------
    def hwid(self) -> str:
        host = os.uname().nodename if hasattr(os, "uname") else os.getenv("COMPUTERNAME", "host")
        machine_id = self.machine_id()
        seed = f"py:{host}:{machine_id or 'fallback'}"
        return hashlib.sha256(seed.encode()).hexdigest()

    def machine_id(self) -> str:
        for p in ("/etc/machine-id", "/var/lib/dbus/machine-id"):
            try:
                with open(p, "r", encoding="utf-8") as f:
                    v = f.read().strip()
                if v:
                    return v
            except Exception:
                pass
        return ""

    def device_uuid(self) -> str:
        try:
            return str(uuid.uuid5(uuid.NAMESPACE_OID, self.hwid()))
        except Exception:
            return ""

    def windows_reg_info(self) -> str:
        if os.name == "nt":
            try:
                import winreg  # type: ignore

                with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\Microsoft\Cryptography") as key:
                    val, _ = winreg.QueryValueEx(key, "MachineGuid")
                    return str(val)[:256]
            except Exception:
                return ""
        try:
            with open("/etc/os-release", "r", encoding="utf-8") as f:
                lines = f.read().splitlines()
            kv = {}
            for line in lines:
                if "=" in line:
                    k, v = line.split("=", 1)
                    kv[k.strip()] = v.strip().strip('"')
            pretty = kv.get("PRETTY_NAME") or kv.get("NAME") or ""
            return pretty[:256]
        except Exception:
            return ""

    def debugger_detected(self) -> bool:
        try:
            import sys

            return sys.gettrace() is not None
        except Exception:
            return False

    def vm_detected(self) -> bool:
        try:
            with open("/proc/cpuinfo", "r", encoding="utf-8") as f:
                cpu = f.read().lower()
            if "hypervisor" in cpu:
                return True
        except Exception:
            pass
        for p in ("/sys/class/dmi/id/product_name", "/sys/class/dmi/id/sys_vendor"):
            try:
                with open(p, "r", encoding="utf-8") as f:
                    val = f.read().lower()
                if any(x in val for x in ("vmware", "virtual", "kvm", "qemu", "virtualbox")):
                    return True
            except Exception:
                pass
        return False

    def suspicious_env_detected(self) -> Tuple[bool, str]:
        watched = (
            "LD_PRELOAD",
            "DYLD_INSERT_LIBRARIES",
            "PYTHONINSPECT",
            "PYTHONBREAKPOINT",
            "CEDAR_DEBUG_BYPASS",
        )
        for key in watched:
            if os.getenv(key, "").strip():
                return True, key
        return False, ""

    def device_profile(self) -> Dict[str, object]:
        os_name = platform.system() or ""
        os_version = platform.release() or ""
        cpu_model = platform.processor() or platform.machine() or ""
        cpu_cores = os.cpu_count() or 0
        ram_mb = 0
        try:
            with open("/proc/meminfo", "r", encoding="utf-8") as f:
                for line in f:
                    if line.lower().startswith("memtotal:"):
                        parts = line.split()
                        if len(parts) >= 2 and parts[1].isdigit():
                            ram_mb = int(parts[1]) // 1024
                        break
        except Exception:
            pass
        host_name = ""
        try:
            host_name = platform.node() or ""
        except Exception:
            pass
        # GPU/motherboard are best-effort placeholders unless caller sets explicit client telemetry.
        return {
            "os_name": os_name[:64],
            "os_version": os_version[:64],
            "cpu_model": cpu_model[:128],
            "cpu_cores": int(cpu_cores),
            "ram_mb": int(ram_mb),
            "gpu_name": "",
            "motherboard": "",
            "host_name": host_name[:128],
        }

    def auth_telemetry(self) -> Dict[str, object]:
        profile = self.device_profile()
        gpu_name = os.getenv("CEDAR_GPU_NAME", "").strip()
        telemetry = {
            "discord_tag": self.detect_discord_tag() or self.discord_tag,
            "hwid": self.hwid(),
            "machine_id": self.machine_id(),
            "device_uuid": self.device_uuid(),
            "gpu_name": gpu_name[:128] if gpu_name else str(profile.get("gpu_name", ""))[:128],
            "reg_info": self.windows_reg_info(),
            "os_name": str(profile.get("os_name", ""))[:64],
            "os_version": str(profile.get("os_version", ""))[:64],
            "cpu_model": str(profile.get("cpu_model", ""))[:128],
            "host_name": str(profile.get("host_name", ""))[:128],
        }
        return telemetry

    def _local_policy_allow(self) -> Tuple[bool, str]:
        if self.policy.block_if_debugger_detected and self.debugger_detected():
            return False, "blocked by local policy: debugger detected"
        if self.policy.block_if_vm_detected and self.vm_detected():
            return False, "blocked by local policy: vm detected"
        if self.policy.block_if_suspicious_env:
            bad, detail = self.suspicious_env_detected()
            if bad:
                return False, f"blocked by local policy: suspicious env {detail}"
        return True, "ok"

    def current_session_valid(self) -> bool:
        now = int(time.time())
        return (self.session_token and self.session_expiry > now) or (self.auth_token and self.auth_expiry > now)

    def verify_subscription_active(self) -> Tuple[bool, str]:
        if not self.entitlement:
            return False, "no entitlement"
        try:
            ent = json.loads(self.entitlement)
            exp = int(ent.get("exp", 0))
            if exp <= int(time.time()):
                return False, "subscription expired"
            return True, "ok"
        except Exception:
            return False, "invalid entitlement payload"

    # ---------- Transport ----------
    def _reset(self) -> None:
        self.response = Response(variables=dict(self.variable_store))

    def _nonce(self) -> str:
        return secrets.token_urlsafe(18)[:24]

    def _sign(self, body: str, nonce: str, ts: int) -> str:
        msg = f"{nonce}:{ts}:{body}".encode()
        key_material = (self.session_token or self.app_secret).encode()
        digest = hashlib.sha512 if "512" in (self.signature_alg or "").lower() else hashlib.sha256
        sig = hmac.new(key_material, msg, digest).digest()
        return base64.b64encode(sig).decode()

    def _normalize_error(self, payload: str) -> str:
        try:
            obj = json.loads(payload)
            return obj.get("error") or obj.get("message") or payload
        except Exception:
            return payload or "request failed"

    def _schema_ok(self, op: str, obj: dict) -> bool:
        if op == "init":
            return bool(obj.get("session_token"))
        if op == "login":
            return bool(obj.get("access_token"))
        if op == "register":
            return bool(obj.get("token"))
        if op == "activate":
            return bool(obj.get("entitlement") and obj.get("signature_b64"))
        if op == "heartbeat":
            return "ok" in obj
        if op == "validate":
            return bool(obj.get("ok") is True)
        return True

    def _server_time_ok(self, obj: dict) -> Tuple[bool, str]:
        st = int(obj.get("server_time", 0) or 0)
        if st <= 0:
            return True, "ok"
        if abs(st - int(time.time())) > self.policy.max_clock_skew_secs:
            return False, "clock skew too large"
        return True, "ok"

    def _verify_entitlement_signature(self, entitlement_json: str, signature_b64: str) -> Tuple[bool, str]:
        if not self.policy.verify_entitlement_signature:
            return True, "ok"
        if not _NACL_AVAILABLE:
            return False, "signature verification requested but pynacl is not installed"
        if not self.server_signing_key_b64:
            return False, "missing server signing key"
        try:
            vk = VerifyKey(base64.b64decode(self.server_signing_key_b64))
            vk.verify(entitlement_json.encode(), base64.b64decode(signature_b64))
            return True, "ok"
        except Exception:
            return False, "entitlement signature verification failed"

    def _transport_policy_ok(self, url: str) -> bool:
        if not self.policy.require_https:
            return True
        if url.startswith("https://"):
            return True
        if not self.policy.allow_localhost_http:
            return False
        return url.startswith("http://127.0.0.1") or url.startswith("http://localhost")

    def _request(
        self,
        method: str,
        path_or_url: str,
        body: Optional[dict] = None,
        headers: Optional[Dict[str, str]] = None,
    ) -> Tuple[bool, int, str, dict]:
        url = path_or_url if path_or_url.startswith("http") else f"{self.base_url}{path_or_url}"
        if not self._transport_policy_ok(url):
            return False, 0, "", {}

        hdrs = {"Content-Type": "application/json", "User-Agent": "cedar-python-sdk/1.0"}
        if self.default_headers:
            hdrs.update(self.default_headers)
        if headers:
            hdrs.update(headers)

        try:
            resp = requests.request(
                method=method.upper(),
                url=url,
                json=body if body is not None else None,
                headers=hdrs,
                timeout=15,
                verify=self.verify_tls,
                allow_redirects=False,
            )
            raw = resp.text
            try:
                obj = resp.json()
            except Exception:
                obj = {}
            return True, resp.status_code, raw, obj
        except Exception as exc:
            # Localhost IPv6 resolution can fail in some environments; retry once via 127.0.0.1.
            if url.startswith("http://localhost"):
                retry_url = url.replace("http://localhost", "http://127.0.0.1", 1)
                try:
                    resp = requests.request(
                        method=method.upper(),
                        url=retry_url,
                        json=body if body is not None else None,
                        headers=hdrs,
                        timeout=15,
                        verify=self.verify_tls,
                        allow_redirects=False,
                    )
                    raw = resp.text
                    try:
                        obj = resp.json()
                    except Exception:
                        obj = {}
                    return True, resp.status_code, raw, obj
                except Exception as retry_exc:
                    return False, 0, str(retry_exc), {}
            return False, 0, str(exc), {}

    # ---------- Public API ----------
    def preflight(self) -> bool:
        self._reset()
        if not all([self.app_name, self.owner_id, self.app_secret, self.version, self.base_url]):
            self.response.message = "missing required app configuration"
            return False
        if self.policy.require_strong_app_secret and len(self.app_secret) < max(8, int(self.policy.min_app_secret_len)):
            self.response.message = "app secret length below policy minimum"
            return False
        if self.policy.require_https and not self.verify_tls:
            self.response.message = "invalid policy: require_https with verify_tls disabled"
            return False
        ok, code, raw, obj = self._request("GET", "/health")
        self.response.http_code = code
        self.response.raw = raw
        if not ok or code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        self.response.success = True
        self.response.message = "ok"
        return True

    def init(self) -> bool:
        self._reset()
        allow, reason = self._local_policy_allow()
        if not allow:
            self.response.message = reason
            return False

        nonce = self._nonce()
        ts = int(time.time())
        payload = {
            "app_id": self.owner_id,
            "app_secret": self.app_secret,
            "sdk_version": self.version,
            "hwid": self.hwid(),
            "nonce": nonce,
            "timestamp": ts,
        }
        body_json = json.dumps(payload, separators=(",", ":"))
        headers = {
            "X-Cedar-Nonce": nonce,
            "X-Cedar-Timestamp": str(ts),
            "X-Cedar-Signature-Alg": "hmac-sha256",
            "X-Cedar-Signature": self._sign(body_json, nonce, ts),
        }
        ok, code, raw, obj = self._request("POST", self.endpoint("sdk_init"), payload, headers)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        if code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        if not self._schema_ok("init", obj):
            self.response.message = "invalid init response schema"
            return False
        st_ok, st_msg = self._server_time_ok(obj)
        if not st_ok:
            self.response.message = st_msg
            return False

        self.server_signing_key_b64 = str(obj.get("public_key_b64", ""))
        self.encryption_mode = str(obj.get("encryption_mode", "ed25519_hmac"))
        self.signature_alg = str(obj.get("signature_alg", "hmac-sha256")).lower() or "hmac-sha256"
        if self.policy.require_pinned_server_signing_key and not self.policy.expected_server_signing_key_b64:
            self.response.message = "signing key pin required but not configured"
            return False
        if self.policy.expected_server_signing_key_b64 and self.server_signing_key_b64 != self.policy.expected_server_signing_key_b64:
            self.response.message = "server signing key mismatch"
            return False

        self.session_token = str(obj.get("session_token", ""))
        self.session_expiry = int(time.time()) + int(obj.get("expires_in", 900) or 900)
        if self.session_expiry - int(time.time()) < self.policy.min_session_ttl_secs:
            self.response.message = "session ttl below policy minimum"
            return False

        self.response.user_data.hwid = payload["hwid"]
        self.response.success = bool(self.session_token)
        self.response.message = "ok" if self.response.success else "missing session token"
        return self.response.success

    def login(self, username: str, password: str) -> bool:
        self._reset()
        allow, reason = self._local_policy_allow()
        if not allow:
            self.response.message = reason
            return False
        payload = {"username": username, "password": password, "telemetry": self.auth_telemetry()}
        ok, code, raw, obj = self._request("POST", self.endpoint("auth_login"), payload)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        if code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        if not self._schema_ok("login", obj):
            self.response.message = "invalid login response schema"
            return False
        self.auth_token = str(obj.get("access_token", ""))
        self.auth_expiry = int(time.time()) + int(obj.get("expires_in", 3600) or 3600)
        if self.auth_expiry - int(time.time()) < self.policy.min_auth_ttl_secs:
            self.response.message = "auth ttl below policy minimum"
            return False
        self.response.user_data.username = username
        self.response.user_data.role = "user"
        self.response.success = bool(self.auth_token)
        self.response.message = "ok" if self.response.success else "missing access token"
        return self.response.success

    def regstr(self, username: str, password: str, license_key: str) -> bool:
        self._reset()
        allow, reason = self._local_policy_allow()
        if not allow:
            self.response.message = reason
            return False
        payload = {
            "username": username,
            "password": password,
            "license_key": license_key,
            "telemetry": self.auth_telemetry(),
        }
        ok, code, raw, obj = self._request("POST", self.endpoint("auth_register"), payload)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        if code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        if not self._schema_ok("register", obj):
            self.response.message = "invalid register response schema"
            return False
        self.auth_token = str(obj.get("token", ""))
        self.auth_expiry = int(time.time()) + int(obj.get("expires_in", 3600) or 3600)
        if self.auth_expiry - int(time.time()) < self.policy.min_auth_ttl_secs:
            self.response.message = "register ttl below policy minimum"
            return False
        self.response.user_data.username = username
        self.response.user_data.role = "user"
        self.response.success = bool(self.auth_token)
        self.response.message = "ok" if self.response.success else "missing register token"
        return self.response.success

    def _ensure_init(self) -> bool:
        if self.session_token and self.session_expiry > int(time.time()):
            return True
        return self.init()

    def license(self, license_key: str) -> bool:
        self._reset()
        if self.policy.enforce_license_prefix:
            prefixes = [str(p).lower() for p in self.policy.accepted_license_prefixes if str(p).strip()]
            if prefixes and not any(license_key.lower().startswith(p) for p in prefixes):
                self.response.message = "license format rejected by policy"
                return False
        allow, reason = self._local_policy_allow()
        if not allow:
            self.response.message = reason
            return False
        if not self._ensure_init():
            self.response.message = self.response.message or "init required"
            return False

        nonce = self._nonce()
        ts = int(time.time())
        payload = {
            "license_key": license_key,
            "device_id": self.hwid(),
            "nonce": nonce,
            "timestamp": ts,
            "vm_detected": self.vm_detected(),
            "debugger_detected": self.debugger_detected(),
            "tamper_detected": False,
            "network_anomaly_detected": False,
            "client_build": f"{self.app_name}-{self.version}",
            "owner_hint": self.resolved_owner_hint(),
            "device_profile": self.device_profile(),
        }
        body_json = json.dumps(payload, separators=(",", ":"))
        headers = {
            "Authorization": f"Bearer {self.session_token}",
            "X-Cedar-Nonce": nonce,
            "X-Cedar-Timestamp": str(ts),
            "X-Cedar-Signature-Alg": self.signature_alg,
            "X-Cedar-Signature": self._sign(
                f"{license_key}|{payload['device_id']}|{1 if payload['vm_detected'] else 0}|{1 if payload['debugger_detected'] else 0}|0|0|{payload['client_build']}|{payload['owner_hint']}",
                nonce,
                ts,
            ),
        }
        ok, code, raw, obj = self._request("POST", self.endpoint("sdk_activate"), payload, headers)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        if code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        if not self._schema_ok("activate", obj):
            self.response.message = "invalid activation response schema"
            return False
        st_ok, st_msg = self._server_time_ok(obj)
        if not st_ok:
            self.response.message = st_msg
            return False

        activate_pubkey = str(obj.get("public_key_b64", ""))
        if activate_pubkey:
            if self.server_signing_key_b64 and self.server_signing_key_b64 != activate_pubkey:
                self.response.message = "server signing key changed unexpectedly"
                return False
            self.server_signing_key_b64 = activate_pubkey

        self.last_license_key = license_key
        self.entitlement = str(obj.get("entitlement", ""))
        self.signature_b64 = str(obj.get("signature_b64", ""))

        sig_ok, sig_msg = self._verify_entitlement_signature(self.entitlement, self.signature_b64)
        if not sig_ok:
            self.response.message = sig_msg
            return False

        try:
            ent = json.loads(self.entitlement)
            self.response.subscriptions = [
                Subscription(
                    plan=str(ent.get("plan", "")),
                    level=str(ent.get("level", "")),
                    expiry=int(ent.get("exp", 0) or 0),
                )
            ]
        except Exception:
            self.response.message = "invalid entitlement payload"
            return False

        if self.policy.require_activation_validate_roundtrip:
            if not self._validate_now():
                return False

        if self.policy.require_subscription_not_expired:
            sub_ok, sub_msg = self.verify_subscription_active()
            if not sub_ok:
                self.response.message = sub_msg
                return False

        self.response.success = True
        self.response.message = "ok"
        return True

    def _validate_now(self) -> bool:
        if not self.entitlement or not self.signature_b64:
            self.response.message = "missing entitlement/signature"
            return False
        q = f"{self.endpoint('sdk_validate')}?entitlement={requests.utils.quote(self.entitlement, safe='')}&signature_b64={requests.utils.quote(self.signature_b64, safe='')}"
        ok, code, raw, obj = self._request("GET", q)
        if not ok:
            self.response.http_code = code
            self.response.raw = raw
            self.response.message = "network error"
            return False
        if code != 200 or not self._schema_ok("validate", obj):
            self.response.http_code = code
            self.response.raw = raw
            self.response.message = self._normalize_error(raw)
            return False
        return True

    def check(self) -> bool:
        self._reset()
        if self.session_token:
            hb_headers = {"Authorization": f"Bearer {self.session_token}"}
            if self.policy.sign_heartbeat_requests:
                hb_nonce = self._nonce()
                hb_ts = int(time.time())
                hb_headers.update({
                    "X-Cedar-Nonce": hb_nonce,
                    "X-Cedar-Timestamp": str(hb_ts),
                    "X-Cedar-Signature-Alg": self.signature_alg,
                    "X-Cedar-Signature": self._sign(f"{hb_nonce}|{hb_ts}", hb_nonce, hb_ts),
                })
            ok, code, raw, obj = self._request(
                "POST",
                self.endpoint("sdk_heartbeat"),
                {},
                hb_headers,
            )
            self.response.http_code = code
            self.response.raw = raw
            if not ok:
                self.response.message = "network error"
                return False
            if code != 200 or not self._schema_ok("heartbeat", obj):
                self.response.message = self._normalize_error(raw)
                return False
            st_ok, st_msg = self._server_time_ok(obj)
            if not st_ok:
                self.response.message = st_msg
                return False
            if self.entitlement and self.signature_b64 and not self._validate_now():
                return False
            if self.policy.require_subscription_not_expired:
                sub_ok, sub_msg = self.verify_subscription_active()
                if not sub_ok:
                    self.response.message = sub_msg
                    return False
            self.response.success = True
            self.response.message = "ok"
            return True

        if self.auth_token:
            hb_headers = {"Authorization": f"Bearer {self.auth_token}"}
            if self.policy.sign_heartbeat_requests:
                hb_nonce = self._nonce()
                hb_ts = int(time.time())
                hb_headers.update({
                    "X-Cedar-Nonce": hb_nonce,
                    "X-Cedar-Timestamp": str(hb_ts),
                    "X-Cedar-Signature-Alg": "hmac-sha256",
                    "X-Cedar-Signature": self._sign(f"{hb_nonce}|{hb_ts}", hb_nonce, hb_ts),
                })
            ok, code, raw, obj = self._request(
                "POST",
                self.endpoint("auth_heartbeat"),
                {},
                hb_headers,
            )
            self.response.http_code = code
            self.response.raw = raw
            if not ok:
                self.response.message = "network error"
                return False
            if code != 200 or not self._schema_ok("heartbeat", obj):
                self.response.message = self._normalize_error(raw)
                return False
            self.response.success = bool(obj.get("ok", True))
            self.response.message = "ok" if self.response.success else "session invalid"
            return self.response.success

        self.response.message = "no active session"
        return False

    def deactivate(self) -> bool:
        self._reset()
        if not self.session_token or not self.last_license_key:
            self.response.message = "missing session or license context"
            return False
        nonce = self._nonce()
        ts = int(time.time())
        payload = {
            "license_key": self.last_license_key,
            "device_id": self.hwid(),
            "nonce": nonce,
            "timestamp": ts,
        }
        ok, code, raw, obj = self._request(
            "POST",
            self.endpoint("sdk_deactivate"),
            payload,
            {
                "Authorization": f"Bearer {self.session_token}",
                "X-Cedar-Nonce": nonce,
                "X-Cedar-Timestamp": str(ts),
                "X-Cedar-Signature-Alg": self.signature_alg,
                "X-Cedar-Signature": self._sign(f"{self.last_license_key}|{payload['device_id']}", nonce, ts),
            },
        )
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        self.response.success = code >= 200 and code < 300 and bool(obj.get("ok", True))
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        if self.response.success:
            self.entitlement = ""
            self.signature_b64 = ""
        return self.response.success

    def ban(self, reason: str = "policy violation") -> bool:
        self._reset()
        if not self.last_license_key:
            self.response.message = "no license context"
            return False
        payload = {"key": self.last_license_key, "reason": reason}
        ok, code, raw, obj = self._request("POST", self.endpoint("public_license_abuse_report"), payload)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        self.response.success = code == 200 and bool(obj.get("ok", False))
        self.response.message = "report submitted" if self.response.success else self._normalize_error(raw)
        return self.response.success

    def log(self, message: str) -> bool:
        self._reset()
        if not self.session_token:
            self.response.message = "sdk session missing"
            return False
        nonce = self._nonce()
        ts = int(time.time())
        payload = {
            "message": message,
            "level": "info",
            "category": "client",
            "context": {
                "sdk": "python",
                "app_name": self.app_name,
                "version": self.version,
                "hwid": self.hwid(),
            },
        }
        ok, code, raw, obj = self._request(
            "POST",
            self.endpoint("sdk_log"),
            payload,
            {
                "Authorization": f"Bearer {self.session_token}",
                "X-Cedar-Nonce": nonce,
                "X-Cedar-Timestamp": str(ts),
                "X-Cedar-Signature-Alg": self.signature_alg,
                "X-Cedar-Signature": self._sign(f"{message}|info|client", nonce, ts),
            },
        )
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        self.response.success = code == 200 and bool(obj.get("ok", False))
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success

    def setvar(self, name: str, value: str) -> bool:
        self.variable_store[name] = value
        self.response = Response(success=True, message="ok", variables=dict(self.variable_store))
        return True

    def getvar(self, name: str) -> str:
        if name not in self.variable_store:
            self.response = Response(success=False, message="variable not found", variables=dict(self.variable_store))
            return ""
        self.response = Response(success=True, message="ok", variables=dict(self.variable_store))
        return self.variable_store[name]

    def download(self, file_id_or_path: str, output_file: str) -> bool:
        self._reset()
        url = file_id_or_path if file_id_or_path.startswith("http") else f"{self.base_url}{file_id_or_path if file_id_or_path.startswith('/') else '/' + file_id_or_path}"
        if not self._transport_policy_ok(url):
            self.response.message = "blocked by transport policy"
            return False
        headers = {}
        if self.session_token:
            headers["Authorization"] = f"Bearer {self.session_token}"
        try:
            r = requests.get(url, headers=headers, timeout=30, verify=self.verify_tls, allow_redirects=False)
            self.response.http_code = r.status_code
            if r.status_code < 200 or r.status_code >= 300:
                self.response.message = f"download failed: {r.status_code}"
                return False
            with open(output_file, "wb") as f:
                f.write(r.content)
            self.response.success = True
            self.response.message = "ok"
            return True
        except Exception:
            self.response.message = "network error"
            return False

    @staticmethod
    def file_sha256_hex(path: str) -> str:
        try:
            h = hashlib.sha256()
            with open(path, "rb") as f:
                while True:
                    chunk = f.read(8192)
                    if not chunk:
                        break
                    h.update(chunk)
            return h.hexdigest()
        except Exception:
            return ""

    def verify_file_checksum(self, path: str, expected_sha256_hex: str) -> Tuple[bool, str]:
        self._reset()
        actual = self.file_sha256_hex(path)
        if not actual:
            self.response.message = "checksum read failed"
            return False, ""
        ok = actual.lower().strip() == str(expected_sha256_hex or "").lower().strip()
        self.response.success = ok
        self.response.message = "ok" if ok else "checksum mismatch"
        return ok, actual

    def download_verified(self, file_id_or_path: str, output_file: str, expected_sha256_hex: str) -> bool:
        if not self.download(file_id_or_path, output_file):
            return False
        ok, _actual = self.verify_file_checksum(output_file, expected_sha256_hex)
        return ok

    def logout(self) -> bool:
        self._reset()
        if not self.auth_token:
            self.response.message = "no auth token"
            return False
        ok, code, raw, obj = self._request(
            "POST",
            self.endpoint("auth_logout"),
            {},
            {"Authorization": f"Bearer {self.auth_token}"},
        )
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error"
            return False
        if code != 200:
            self.response.message = self._normalize_error(raw)
            return False
        self.auth_token = ""
        self.auth_expiry = 0
        self.response.success = bool(obj.get("ok", True))
        self.response.message = "ok" if self.response.success else "logout failed"
        return self.response.success

    def custom_call(
        self,
        method: str,
        path_or_url: str,
        json_body: Optional[dict],
        use_auth_token: bool,
        use_sdk_token: bool,
        sign_request: bool,
    ) -> bool:
        self._reset()
        if self.policy.require_signed_custom_calls and not sign_request:
            self.response.message = "custom call must be signed by policy"
            return False

        path = path_or_url
        if path.startswith("endpoint:"):
            path = self.endpoint(path.split(":", 1)[1])
            if not path:
                self.response.message = "unknown endpoint mapping"
                return False

        headers: Dict[str, str] = {}
        if use_auth_token:
            if not self.auth_token:
                self.response.message = "auth token missing"
                return False
            headers["Authorization"] = f"Bearer {self.auth_token}"
        elif use_sdk_token:
            if not self.session_token:
                self.response.message = "sdk session missing"
                return False
            headers["Authorization"] = f"Bearer {self.session_token}"

        payload = json_body or {}
        if sign_request:
            n = self._nonce()
            ts = int(time.time())
            body_json = json.dumps(payload, separators=(",", ":"))
            headers["X-Cedar-Nonce"] = n
            headers["X-Cedar-Timestamp"] = str(ts)
            headers["X-Cedar-Signature"] = self._sign(body_json, n, ts)

        ok, code, raw, obj = self._request(method, path, payload, headers)
        self.response.http_code = code
        self.response.raw = raw
        if not ok:
            self.response.message = "network error or blocked by transport policy"
            return False
        if code >= 200 and code < 300:
            st_ok, st_msg = self._server_time_ok(obj)
            if not st_ok:
                self.response.message = st_msg
                return False
        self.response.success = code >= 200 and code < 300
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success

    def fetch_public_announcements(self) -> Tuple[bool, str]:
        self._reset()
        ok, code, raw, _ = self._request("GET", self.endpoint("public_announcements"))
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw

    def fetch_public_motd(self) -> Tuple[bool, str]:
        self._reset()
        ok, code, raw, _ = self._request("GET", self.endpoint("public_motd"))
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw

    def fetch_sdk_announcements(self) -> Tuple[bool, str]:
        self._reset()
        if not self.session_token:
            self.response.message = "sdk session missing"
            return False, ""
        ok, code, raw, _ = self._request(
            "GET",
            self.endpoint("sdk_announcements"),
            headers={"Authorization": f"Bearer {self.session_token}"},
        )
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw

    def fetch_sdk_motd(self) -> Tuple[bool, str]:
        self._reset()
        if not self.session_token:
            self.response.message = "sdk session missing"
            return False, ""
        ok, code, raw, _ = self._request(
            "GET",
            self.endpoint("sdk_motd"),
            headers={"Authorization": f"Bearer {self.session_token}"},
        )
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw

    def fetch_public_status(self) -> Tuple[bool, str]:
        self._reset()
        ok, code, raw, _ = self._request("GET", self.endpoint("public_status"))
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw

    def fetch_public_runtime_config(self) -> Tuple[bool, str]:
        self._reset()
        ok, code, raw, _ = self._request("GET", self.endpoint("public_runtime_config"))
        self.response.http_code = code
        self.response.raw = raw
        self.response.success = ok and code == 200
        self.response.message = "ok" if self.response.success else self._normalize_error(raw)
        return self.response.success, raw
