# CEDAR Python SDK

Python SDK reference implementation aligned with CEDAR C/C++ integration model.

## Features
- API bootstrap / auth / license lifecycle
- Signed SDK requests
- Encryption/signature negotiation support from `/api/sdk/init`
- Entitlement verification hooks
- Device + telemetry helpers
- Checksum helpers for secure download verification

## API Surface
- `preflight()`
- `init()`
- `login()`
- `regstr()`
- `license()`
- `check()`
- `deactivate()`
- `logout()`
- `log()`
- `setvar()` / `getvar()`
- `download()` / `download_verified()`
- `verify_file_checksum()` / `file_sha256_hex()`
- `custom_call()`
- `fetch_public_runtime_config()`

## Negotiated Crypto Profile
After `init()`, SDK stores:
- `sdk.encryption_mode`
- `sdk.signature_alg`

SDK then uses negotiated algorithm when signing SDK request flows.

## Install
```bash
cd /home/admin/license_cloud_auth/cedar_python_sdk
python3 -m pip install -r requirements.txt
```

## Examples
- Interactive example: `examples/example.py`
- Env-driven example: `examples/env_example.py`
- Both examples now use structured, colorized console output for easier operator readability.

Run:
```bash
cd /home/admin/license_cloud_auth/cedar_python_sdk/examples
python3 example.py
```

## Minimal Example
```python
from cedar_sdk import CedarSDK, SecurityPolicy

sdk = CedarSDK(
    app_name="MyApp",
    owner_id="app_xxx",
    app_secret="secret_xxx",
    version="1.0.0",
    base_url="https://www.cedarcloud.app",
)

policy = SecurityPolicy()
policy.verify_entitlement_signature = True
sdk.set_security_policy(policy)

assert sdk.preflight(), sdk.response.message
assert sdk.init(), sdk.response.message
print(sdk.encryption_mode, sdk.signature_alg)
assert sdk.license("LIC-XXXX"), sdk.response.message
assert sdk.check(), sdk.response.message
```

## Documentation
- `docs/INTEGRATION_GUIDE.md`
- `docs/SECURITY_GUIDE.md`
- `docs/API_ALIGNMENT.md`

## Notes
- For local testing behind strict IP controls, set forwarded headers with `set_default_headers`.
- Use HTTPS + TLS verification in production.
- Keep app secrets outside shipped binaries whenever possible.
