<?php
namespace Cedar;

class CedarClient {
    private string $baseUrl;
    private string $appId;
    private string $appSecret;
    private string $sdkVersion;
    public string $token = '';
    public string $sdkToken = '';
    public string $entitlement = '';
    public string $signatureB64 = '';

    public function __construct(string $baseUrl, string $appId, string $appSecret, string $sdkVersion = '1.0.0') {
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->appId = $appId;
        $this->appSecret = $appSecret;
        $this->sdkVersion = $sdkVersion;
    }

    private function request(string $path, string $method = 'GET', ?array $body = null, string $bearer = ''): array {
        $ch = curl_init($this->baseUrl . $path);
        $headers = ['Content-Type: application/json'];
        if ($bearer !== '') $headers[] = 'Authorization: Bearer ' . $bearer;
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        if ($body !== null) curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        $raw = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        $payload = json_decode($raw ?: '{}', true);
        if ($code < 200 || $code >= 300) {
            $msg = $payload['error'] ?? ('HTTP ' . $code);
            throw new \RuntimeException($msg);
        }
        return is_array($payload) ? $payload : [];
    }

    public function health(): array { return $this->request('/health'); }
    public function login(string $username, string $password): array {
        $out = $this->request('/api/auth/login', 'POST', ['username' => $username, 'password' => $password]);
        $this->token = $out['access_token'] ?? '';
        return $out;
    }
    public function heartbeat(): array { return $this->request('/api/auth/heartbeat', 'POST', [], $this->token); }

    public function sdkInit(string $deviceId, string $hwid = ''): array {
        $out = $this->request('/api/sdk/init', 'POST', [
            'app_id' => $this->appId,
            'app_secret' => $this->appSecret,
            'sdk_version' => $this->sdkVersion,
            'device_id' => $deviceId,
            'hwid' => $hwid,
            'nonce' => substr(bin2hex(random_bytes(16)), 0, 24),
            'timestamp' => time(),
        ]);
        $this->sdkToken = $out['token'] ?? '';
        return $out;
    }

    public function activateLicense(string $licenseKey, string $deviceId, string $hwid = ''): array {
        $out = $this->request('/api/sdk/license/activate', 'POST', [
            'license_key' => $licenseKey,
            'device_id' => $deviceId,
            'hwid' => $hwid,
            'nonce' => substr(bin2hex(random_bytes(16)), 0, 24),
            'timestamp' => time(),
        ], $this->sdkToken);
        $this->entitlement = $out['entitlement'] ?? '';
        $this->signatureB64 = $out['signature_b64'] ?? '';
        return $out;
    }

    public function sdkHeartbeat(): array { return $this->request('/api/sdk/heartbeat', 'POST', [], $this->sdkToken); }
    public function validateEntitlement(): array {
        $q = http_build_query(['entitlement' => $this->entitlement, 'signature_b64' => $this->signatureB64]);
        return $this->request('/api/sdk/license/validate?' . $q);
    }
}
