export class CedarClient {
  constructor(config) {
    this.baseUrl = (config.baseUrl || '').replace(/\/$/, '');
    this.appId = config.appId || '';
    this.appSecret = config.appSecret || '';
    this.sdkVersion = config.sdkVersion || '1.0.0';
    this.token = '';
    this.sdkToken = '';
    this.entitlement = '';
    this.signatureB64 = '';
  }

  async request(path, method = 'GET', body = null, authToken = '') {
    const headers = { 'Content-Type': 'application/json' };
    if (authToken) headers.Authorization = `Bearer ${authToken}`;
    const res = await fetch(`${this.baseUrl}${path}`, {
      method,
      headers,
      body: body ? JSON.stringify(body) : undefined,
    });
    const payload = await res.json().catch(() => ({}));
    if (!res.ok) throw new Error(payload.error || `HTTP ${res.status}`);
    return payload;
  }

  async health() { return this.request('/health'); }
  async register(username, password, email = '') {
    const out = await this.request('/api/auth/register', 'POST', { username, password, email });
    this.token = out.token || '';
    return out;
  }
  async login(username, password) {
    const out = await this.request('/api/auth/login', 'POST', { username, password });
    this.token = out.access_token || '';
    return out;
  }
  async heartbeat() { return this.request('/api/auth/heartbeat', 'POST', {}, this.token); }

  async sdkInit(deviceId, hwid = '') {
    const out = await this.request('/api/sdk/init', 'POST', {
      app_id: this.appId,
      app_secret: this.appSecret,
      sdk_version: this.sdkVersion,
      device_id: deviceId,
      hwid,
      nonce: crypto.randomUUID().replace(/-/g, '').slice(0, 24),
      timestamp: Math.floor(Date.now() / 1000),
    });
    this.sdkToken = out.token || '';
    return out;
  }

  async activateLicense(licenseKey, deviceId, hwid = '') {
    const out = await this.request('/api/sdk/license/activate', 'POST', {
      license_key: licenseKey,
      device_id: deviceId,
      hwid,
      nonce: crypto.randomUUID().replace(/-/g, '').slice(0, 24),
      timestamp: Math.floor(Date.now() / 1000),
    }, this.sdkToken);
    this.entitlement = out.entitlement || '';
    this.signatureB64 = out.signature_b64 || '';
    return out;
  }

  async sdkHeartbeat() { return this.request('/api/sdk/heartbeat', 'POST', {}, this.sdkToken); }
  async validateEntitlement() {
    const q = new URLSearchParams({ entitlement: this.entitlement, signature_b64: this.signatureB64 }).toString();
    return this.request(`/api/sdk/license/validate?${q}`);
  }
}
