#include "cedar_sdk/cedar_sdk_c.h"

#include "cedar_sdk/cedar_sdk.hpp"

#include <memory>
#include <string>

struct cedar_sdk_handle {
  std::unique_ptr<cedar::api> impl;
  std::string tmp;
};

namespace {

const char* to_cstr(std::string& storage, const std::string& value) {
  storage = value;
  return storage.c_str();
}

const char* safe(const char* s) {
  return s ? s : "";
}

}  // namespace

extern "C" {

cedar_sdk_handle* cedar_sdk_create(const char* app_name,
                                   const char* owner_id,
                                   const char* app_secret,
                                   const char* version,
                                   const char* base_url,
                                   int verify_tls) {
  auto* h = new cedar_sdk_handle();
  h->impl = std::make_unique<cedar::api>(safe(app_name),
                                         safe(owner_id),
                                         safe(app_secret),
                                         safe(version),
                                         safe(base_url),
                                         verify_tls != 0);
  return h;
}

void cedar_sdk_destroy(cedar_sdk_handle* handle) {
  delete handle;
}

int cedar_sdk_init(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->init()) ? 1 : 0;
}

int cedar_sdk_preflight(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->preflight()) ? 1 : 0;
}

int cedar_sdk_login(cedar_sdk_handle* handle, const char* username, const char* password) {
  return (handle && handle->impl && handle->impl->login(safe(username), safe(password))) ? 1 : 0;
}

int cedar_sdk_regstr(cedar_sdk_handle* handle,
                     const char* username,
                     const char* password,
                     const char* license_key) {
  return (handle && handle->impl &&
          handle->impl->regstr(safe(username), safe(password), safe(license_key)))
             ? 1
             : 0;
}

int cedar_sdk_license(cedar_sdk_handle* handle, const char* license_key) {
  return (handle && handle->impl && handle->impl->license(safe(license_key))) ? 1 : 0;
}

int cedar_sdk_deactivate(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->deactivate()) ? 1 : 0;
}

int cedar_sdk_check(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->check()) ? 1 : 0;
}

int cedar_sdk_ban(cedar_sdk_handle* handle, const char* reason) {
  return (handle && handle->impl && handle->impl->ban(safe(reason))) ? 1 : 0;
}

int cedar_sdk_log(cedar_sdk_handle* handle, const char* message) {
  return (handle && handle->impl && handle->impl->log(safe(message))) ? 1 : 0;
}

int cedar_sdk_setvar(cedar_sdk_handle* handle, const char* name, const char* value) {
  return (handle && handle->impl && handle->impl->setvar(safe(name), safe(value))) ? 1 : 0;
}

const char* cedar_sdk_getvar(cedar_sdk_handle* handle, const char* name) {
  if (!handle || !handle->impl) return "";
  return to_cstr(handle->tmp, handle->impl->getvar(safe(name)));
}

int cedar_sdk_download(cedar_sdk_handle* handle, const char* file_id_or_path, const char* output_file) {
  return (handle && handle->impl &&
          handle->impl->download(safe(file_id_or_path), safe(output_file)))
             ? 1
             : 0;
}

int cedar_sdk_logout(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->logout()) ? 1 : 0;
}

int cedar_sdk_fetch_public_announcements(cedar_sdk_handle* handle) {
  std::string out;
  return (handle && handle->impl && handle->impl->fetch_public_announcements(&out)) ? 1 : 0;
}

int cedar_sdk_fetch_sdk_announcements(cedar_sdk_handle* handle) {
  std::string out;
  return (handle && handle->impl && handle->impl->fetch_sdk_announcements(&out)) ? 1 : 0;
}

int cedar_sdk_fetch_public_motd(cedar_sdk_handle* handle) {
  std::string out;
  return (handle && handle->impl && handle->impl->fetch_public_motd(&out)) ? 1 : 0;
}

int cedar_sdk_fetch_sdk_motd(cedar_sdk_handle* handle) {
  std::string out;
  return (handle && handle->impl && handle->impl->fetch_sdk_motd(&out)) ? 1 : 0;
}

int cedar_sdk_fetch_public_status(cedar_sdk_handle* handle) {
  std::string out;
  return (handle && handle->impl && handle->impl->fetch_public_status(&out)) ? 1 : 0;
}

int cedar_sdk_current_session_valid(cedar_sdk_handle* handle) {
  return (handle && handle->impl && handle->impl->current_session_valid()) ? 1 : 0;
}

int cedar_sdk_verify_subscription_active(cedar_sdk_handle* handle) {
  std::string reason;
  return (handle && handle->impl && handle->impl->verify_subscription_active(&reason)) ? 1 : 0;
}

int cedar_sdk_custom_call(cedar_sdk_handle* handle,
                          const char* method,
                          const char* path_or_url,
                          const char* json_body,
                          int use_auth_token,
                          int use_sdk_token,
                          int sign_request) {
  return (handle && handle->impl &&
          handle->impl->custom_call(safe(method),
                                    safe(path_or_url),
                                    safe(json_body),
                                    use_auth_token != 0,
                                    use_sdk_token != 0,
                                    sign_request != 0))
             ? 1
             : 0;
}

void cedar_sdk_set_endpoint(cedar_sdk_handle* handle, const char* name, const char* path) {
  if (!handle || !handle->impl) return;
  handle->impl->set_endpoint(safe(name), safe(path));
}

void cedar_sdk_set_security_policy(cedar_sdk_handle* handle,
                                   int require_https,
                                   int allow_localhost_http,
                                   long max_clock_skew_secs) {
  if (!handle || !handle->impl) return;
  cedar::security_policy_t policy;
  policy.require_https = require_https != 0;
  policy.allow_localhost_http = allow_localhost_http != 0;
  policy.max_clock_skew_secs = max_clock_skew_secs;
  handle->impl->set_security_policy(policy);
}

void cedar_sdk_set_signing_key_pin(cedar_sdk_handle* handle,
                                   const char* expected_key_b64,
                                   int require_pin) {
  if (!handle || !handle->impl) return;
  handle->impl->set_signing_key_pin(safe(expected_key_b64), require_pin != 0);
}

void cedar_sdk_set_local_guards(cedar_sdk_handle* handle,
                                int block_debugger,
                                int block_vm,
                                int block_blacklist) {
  if (!handle || !handle->impl) return;
  handle->impl->set_local_guards(block_debugger != 0, block_vm != 0, block_blacklist != 0);
}

int cedar_sdk_load_profile_from_env(cedar_sdk_handle* handle, const char* env_prefix) {
  return (handle && handle->impl && handle->impl->load_profile_from_env(safe(env_prefix))) ? 1 : 0;
}

int cedar_sdk_load_profile_from_encrypted_blob(cedar_sdk_handle* handle,
                                               const char* profile_blob_b64,
                                               const char* decrypt_secret) {
  return (handle && handle->impl &&
          handle->impl->load_profile_from_encrypted_blob(safe(profile_blob_b64), safe(decrypt_secret)))
             ? 1
             : 0;
}

const char* cedar_sdk_last_message(cedar_sdk_handle* handle) {
  if (!handle || !handle->impl) return "";
  return to_cstr(handle->tmp, handle->impl->response().message);
}

long cedar_sdk_last_http_code(cedar_sdk_handle* handle) {
  if (!handle || !handle->impl) return 0;
  return handle->impl->response().http_code;
}

const char* cedar_sdk_last_raw(cedar_sdk_handle* handle) {
  if (!handle || !handle->impl) return "";
  return to_cstr(handle->tmp, handle->impl->response().raw);
}

}  // extern "C"
