#pragma once

#include <cstdint>
#include <map>
#include <string>
#include <unordered_map>
#include <vector>

namespace cedar {

struct user_data_t {
  std::string username;
  std::string role;
  std::string hwid;
  std::string ip;
};

struct subscription_t {
  std::string plan;
  std::string level;
  std::int64_t expiry = 0;
};

struct response_t {
  bool success = false;
  std::string message;
  long http_code = 0;
  std::string raw;
  user_data_t user_data;
  std::vector<subscription_t> subscriptions;
  std::map<std::string, std::string> variables;
};

struct security_policy_t {
  bool require_https = false;
  bool allow_localhost_http = true;
  bool require_strong_app_secret = true;
  std::int64_t min_app_secret_len = 24;
  std::int64_t max_clock_skew_secs = 180;
  bool verify_entitlement_signature = true;
  bool require_pinned_server_signing_key = false;
  std::string expected_server_signing_key_b64;
  std::string tls_pinned_public_key;
  bool block_if_debugger_detected = false;
  bool block_if_vm_detected = false;
  bool block_if_suspicious_env = false;
  bool block_if_blacklisted_process = false;
  bool enforce_license_prefix = false;
  bool sign_heartbeat_requests = true;
  bool require_activation_validate_roundtrip = true;
  bool require_subscription_not_expired = true;
  std::int64_t min_session_ttl_secs = 120;
  std::int64_t min_auth_ttl_secs = 120;
  bool require_signed_custom_calls = false;
  std::vector<std::string> accepted_license_prefixes = {"LIC-", "CEDAR-"};
  std::vector<std::string> process_blacklist = {
      "ida", "x64dbg", "ollydbg", "wireshark", "fiddler"};
};

class api {
 public:
  api(std::string app_name,
      std::string owner_id,
      std::string app_secret,
      std::string version,
      std::string base_url,
      bool verify_tls = true);
  ~api();

  void set_security_policy(const security_policy_t& policy);
  void set_signing_key_pin(const std::string& expected_key_b64, bool require_pin);
  void set_local_guards(bool block_debugger, bool block_vm, bool block_blacklist);
  void set_endpoint(const std::string& name, const std::string& path);
  std::string endpoint(const std::string& name) const;
  bool preflight();

  bool init();
  bool login(const std::string& username, const std::string& password);
  bool regstr(const std::string& username, const std::string& password, const std::string& license_key);
  bool license(const std::string& license_key);
  bool check();
  bool ban(const std::string& reason = "policy violation");
  bool log(const std::string& message);
  bool setvar(const std::string& name, const std::string& value);
  std::string getvar(const std::string& name);
  bool download(const std::string& file_id_or_path, const std::string& output_file);
  bool verify_file_checksum(const std::string& path,
                            const std::string& expected_sha256_hex,
                            std::string* actual_hex = nullptr);
  bool download_verified(const std::string& file_id_or_path,
                         const std::string& output_file,
                         const std::string& expected_sha256_hex,
                         std::string* actual_hex = nullptr);
  bool logout();
  bool custom_call(const std::string& method,
                   const std::string& path_or_url,
                   const std::string& json_body,
                   bool use_auth_token,
                   bool use_sdk_token,
                   bool sign_request);
  bool deactivate();
  bool fetch_public_announcements(std::string* raw_json);
  bool fetch_sdk_announcements(std::string* raw_json);
  bool fetch_public_motd(std::string* raw_json);
  bool fetch_sdk_motd(std::string* raw_json);
  bool fetch_public_status(std::string* raw_json);
  bool fetch_public_runtime_config(std::string* raw_json);
  bool verify_subscription_active(std::string* reason) const;
  bool current_session_valid() const;

  const response_t& response() const;
  const std::string& session_id() const;
  const std::string& auth_token() const;
  std::string hwid() const;
  const std::string& encryption_mode() const;
  const std::string& signature_algorithm() const;
  bool debugger_detected() const;
  bool vm_detected() const;
  bool suspicious_env_detected(std::string* detail = nullptr) const;
  bool process_blacklist_hit(std::string* matched_name) const;
  bool load_profile_from_env(const std::string& env_prefix);
  bool load_profile_from_encrypted_blob(const std::string& profile_blob_b64,
                                        const std::string& decrypt_secret);

  static std::string encrypt(const std::string& plaintext, const std::string& secret);
  static std::string decrypt(const std::string& blob_b64, const std::string& secret);
  static std::string file_sha256_hex(const std::string& path);

 private:
  struct http_result_t {
    bool ok = false;
    long status = 0;
    std::string body;
  };

  std::string app_name_;
  std::string owner_id_;
  std::string app_secret_;
  std::string version_;
  std::string base_url_;
  bool verify_tls_ = true;

  std::string session_token_;
  std::string auth_token_;
  std::int64_t session_expiry_ = 0;
  std::int64_t auth_expiry_ = 0;
  std::string entitlement_;
  std::string signature_;
  std::string active_encryption_mode_ = "ed25519_hmac";
  std::string active_signature_alg_ = "hmac-sha256";
  std::string last_license_key_;
  std::string server_signing_key_b64_;
  std::map<std::string, std::string> variable_store_;
  std::unordered_map<std::string, std::string> endpoints_;
  security_policy_t policy_;

  response_t response_;

  bool license_state_from_entitlement(const std::string& entitlement_json);
  bool ensure_init();
  bool transport_policy_ok(const std::string& url_or_path) const;
  bool update_server_time_guard(const std::string& body);
  bool validate_response_schema(const std::string& op, const std::string& body);
  bool verify_entitlement_signature(const std::string& entitlement_json,
                                    const std::string& signature_b64,
                                    std::string* error) const;
  bool local_policy_allow(std::string* reason) const;
  static void secure_clear(std::string* value);
  void reset_response();

  static std::int64_t now();
  static std::string nonce();
  static std::string json_escape(const std::string& value);
  static std::string json_get_string(const std::string& json, const std::string& key);
  static std::int64_t json_get_int(const std::string& json, const std::string& key, std::int64_t fallback = 0);
  static bool json_get_bool(const std::string& json, const std::string& key, bool fallback = false);
  static std::string normalize_error(const std::string& payload);
  std::string sign(const std::string& body, const std::string& nonce, std::int64_t ts) const;

  http_result_t http_json(const std::string& method,
                          const std::string& path,
                          const std::string& body,
                          const std::vector<std::string>& headers) const;
  http_result_t http_download(const std::string& file_id_or_path,
                              const std::string& output_file,
                              const std::vector<std::string>& headers) const;
};

}  // namespace cedar
