#include "cedar_sdk/cedar_sdk.hpp"

#include <cstdlib>
#include <iostream>
#include <string>

namespace {

bool use_color() {
#if defined(_WIN32)
  return false;
#else
  return std::getenv("NO_COLOR") == nullptr;
#endif
}

std::string paint(const std::string& text, const char* code) {
  if (!use_color()) return text;
  return std::string(code) + text + "\033[0m";
}

void header(const std::string& title) {
  std::cout << paint("\n================================================================", "\033[90m") << "\n";
  std::cout << paint(title, "\033[36;1m") << "\n";
  std::cout << paint("================================================================", "\033[90m") << "\n";
}

void ok(const std::string& msg) { std::cout << paint("[OK] ", "\033[32m") << msg << "\n"; }
void warn(const std::string& msg) { std::cout << paint("[WARN] ", "\033[33m") << msg << "\n"; }
void fail(const std::string& msg) { std::cerr << paint("[FAIL] ", "\033[31m") << msg << "\n"; }

void kv(const std::string& k, const std::string& v) {
  std::cout << paint("  " + k, "\033[90m") << ": " << v << "\n";
}

}  // namespace

int main() {
  cedar::api auth(
      "CEDAR",
      "app_your_owner_id",
      "your_app_secret",
      "1.0.0",
      "https://www.cedarcloud.app",
      false);

  auth.load_profile_from_env("CEDAR");

  cedar::security_policy_t policy;
  policy.require_https = false;
  policy.allow_localhost_http = true;
  policy.max_clock_skew_secs = 180;
  policy.verify_entitlement_signature = true;
  policy.block_if_debugger_detected = false;
  policy.block_if_vm_detected = false;
  policy.block_if_blacklisted_process = false;
  auth.set_security_policy(policy);

  header("CEDAR C++ SDK Example");

  if (!auth.preflight()) {
    fail("preflight failed: " + auth.response().message);
    return 1;
  }
  ok("preflight passed");

  if (!auth.init()) {
    fail("init failed: " + auth.response().message);
    return 1;
  }
  ok("session initialized");
  kv("encryption_mode", auth.encryption_mode());
  kv("signature_alg", auth.signature_algorithm());
  kv("session_valid", auth.current_session_valid() ? "yes" : "no");

  std::string runtime_config;
  if (auth.fetch_public_runtime_config(&runtime_config)) {
    ok("runtime config synced from API");
    kv("runtime_config_bytes", std::to_string(runtime_config.size()));
  } else {
    warn("runtime config unavailable: " + auth.response().message);
  }

  std::string key;
  std::cout << paint("license key", "\033[36m") << ": ";
  std::getline(std::cin, key);

  if (!auth.license(key)) {
    fail("license failed: " + auth.response().message);
    return 1;
  }
  ok("license activation succeeded");

  if (!auth.check()) {
    fail("check failed: " + auth.response().message);
    return 1;
  }
  ok("heartbeat check passed");

  std::string sub_reason;
  const bool sub_ok = auth.verify_subscription_active(&sub_reason);
  kv("subscription_active", std::string(sub_ok ? "yes" : "no") + " (" + (sub_reason.empty() ? "ok" : sub_reason) + ")");

  if (const char* emit_log = std::getenv("CEDAR_EXAMPLE_EMIT_LOG"); emit_log && std::string(emit_log) == "1") {
    auth.log("example client connected and validated");
  }

  auth.custom_call("POST", "endpoint:sdk_heartbeat", "{}", false, true, true);
  auth.custom_call("GET", "/health", "", false, false, false);

  std::string public_status;
  auth.fetch_public_status(&public_status);
  std::string public_announcements;
  auth.fetch_public_announcements(&public_announcements);
  std::string sdk_announcements;
  auth.fetch_sdk_announcements(&sdk_announcements);
  std::string public_motd;
  auth.fetch_public_motd(&public_motd);

  std::string sdk_motd;
  if (auth.fetch_sdk_motd(&sdk_motd)) {
    ok("sdk motd fetched");
    kv("sdk_motd_bytes", std::to_string(sdk_motd.size()));
  }

  kv("public_status_bytes", std::to_string(public_status.size()));
  kv("public_announcements_bytes", std::to_string(public_announcements.size()));
  kv("sdk_announcements_bytes", std::to_string(sdk_announcements.size()));
  kv("public_motd_bytes", std::to_string(public_motd.size()));

  auth.setvar("last_user", "demo");
  kv("getvar(last_user)", auth.getvar("last_user"));

  if (const char* dl = std::getenv("CEDAR_EXAMPLE_DOWNLOAD_URL")) {
    if (const char* sum = std::getenv("CEDAR_EXAMPLE_DOWNLOAD_SHA256")) {
      std::string actual;
      const bool dl_ok = auth.download_verified(dl, "./download.bin", sum, &actual);
      if (dl_ok) {
        ok("download + checksum verified");
        kv("download_sha256", actual);
      } else {
        fail("download verification failed: " + auth.response().message);
        return 1;
      }
    }
  } else {
    warn("download verification skipped (set CEDAR_EXAMPLE_DOWNLOAD_URL + CEDAR_EXAMPLE_DOWNLOAD_SHA256)");
  }

  if (const char* deact = std::getenv("CEDAR_EXAMPLE_DEACTIVATE"); deact && std::string(deact) == "1") {
    kv("deactivate", auth.deactivate() ? "ok" : auth.response().message);
  }
  if (const char* out = std::getenv("CEDAR_EXAMPLE_LOGOUT"); out && std::string(out) == "1") {
    kv("logout", auth.logout() ? "ok" : auth.response().message);
  }

  ok("example completed successfully");
  return 0;
}
