# CEDAR SDK Library (C/C++)

C/C++ SDK library for integrating native applications with CEDAR.

## What This SDK Provides
- Session bootstrap via `/api/sdk/init`
- License activation/deactivation flow
- Heartbeat/session validation
- Signed request support with algorithm negotiation
- Entitlement signature verification (Ed25519)
- Checksum verification helpers for secure downloads
- C ABI wrapper for C projects and loader-style integrations

## Primary Class
`cedar::api`

Core methods:
- `preflight()`
- `init()`
- `login()`
- `regstr()`
- `license()`
- `check()`
- `deactivate()`
- `logout()`
- `log()`
- `setvar()` / `getvar()`
- `download()` / `download_verified()`
- `verify_file_checksum()` / `file_sha256_hex()`
- `custom_call()`
- `fetch_public_runtime_config()`

## Encryption and Signing Negotiation
After `init()`, server returns:
- `encryption_mode`
- `signature_alg`
- `request_signing_required`

SDK stores these and signs SDK requests with the negotiated algorithm.

Accessors:
- `encryption_mode()`
- `signature_algorithm()`

## Signed SDK Endpoints (Handled by SDK)
- `POST /api/sdk/license/activate`
- `POST /api/sdk/heartbeat`
- `POST /api/sdk/license/deactivate`
- `POST /api/sdk/log`

## Security Policy Controls
`security_policy_t` includes:
- TLS/HTTPS enforcement options
- Signing key pin options
- Session/auth TTL minimums
- Entitlement signature verification
- Anti-debug / anti-VM / blacklist process guards
- Optional signed custom calls requirement

## Build
```bash
cd /home/admin/license_cloud_auth/cedar_sdk_library
cmake -S . -B build
cmake --build build -j
```

## Install
```bash
cmake -S . -B build -DCMAKE_INSTALL_PREFIX=/usr/local
cmake --build build -j
cmake --install build
```

## CMake Consume
```cmake
find_package(CedarSDK CONFIG REQUIRED)
target_link_libraries(my_cpp_target PRIVATE cedar::cedar_sdk_cpp)
target_link_libraries(my_c_target PRIVATE cedar::cedar_sdk_c)
```

## Example Usage (C++)
```cpp
cedar::api sdk("MyApp", "app_id", "app_secret", "1.0.0", "https://api.example.com", true);

cedar::security_policy_t policy;
policy.require_https = true;
policy.verify_entitlement_signature = true;
sdk.set_security_policy(policy);

if (!sdk.preflight()) return;
if (!sdk.init()) return;

// negotiated profile from server
std::string mode = sdk.encryption_mode();
std::string alg = sdk.signature_algorithm();

if (!sdk.license("LIC-XXXX")) return;
if (!sdk.check()) return;

std::string actual;
bool ok = sdk.download_verified("/static/protected.bin", "./protected.bin", "<expected_sha256>", &actual);
```

## C API Wrapper
Header: `include/cedar_sdk/cedar_sdk_c.h`

Includes wrappers for:
- lifecycle methods
- custom call
- endpoint remap
- security policy setup
- key pinning
- encrypted profile loading
- announcement/status fetch

## Documentation
- `docs/INTEGRATION_GUIDE.md`
- `docs/SECURITY_GUIDE.md`
- `docs/API_ALIGNMENT.md`
- `examples/cpp_example.cpp`
