#pragma once

#include <cstdint>
#include <map>
#include <string>
#include <vector>

namespace cedar {

struct ClientConfig {
  std::string base_url;
  std::string app_name;
  std::string owner_id;
  std::string app_secret;
  std::string version = "1.0.0";
  bool verify_tls = true;
  long timeout_ms = 15000;
  // Optional pinned public key file/path for libcurl (PEM/DER hash format accepted by curl).
  std::string tls_pinned_public_key;
  // Optional backend signing key pin (from /api/sdk/init public_key_b64).
  std::string expected_server_signing_key_b64;
  // Max allowed drift between client clock and server_time (seconds).
  std::int64_t max_clock_skew_secs = 180;
  // Local defensive policy controls before entitlement activation.
  bool block_if_debugger_detected = false;
  bool block_if_vm_detected = false;
  bool block_if_blacklisted_process_detected = false;
  std::vector<std::string> process_blacklist = {
      "ida", "x64dbg", "ollydbg", "wireshark", "fiddler"};
};

struct ApiResponse {
  bool ok = false;
  long http_status = 0;
  std::string message;
  std::string raw;
  std::map<std::string, std::string> fields;
};

struct SessionState {
  std::string sdk_session_token;
  std::string auth_token;
  std::int64_t auth_expires_at = 0;
  std::int64_t sdk_expires_at = 0;
  bool licensed = false;
  std::string entitlement_b64;
  std::string signature_b64;
  std::string server_public_key_b64;
};

class SecureString {
 public:
  SecureString() = default;
  explicit SecureString(std::string value);
  ~SecureString();

  SecureString(const SecureString&) = delete;
  SecureString& operator=(const SecureString&) = delete;

  SecureString(SecureString&& other) noexcept;
  SecureString& operator=(SecureString&& other) noexcept;

  const std::string& str() const;
  bool empty() const;
  void clear();

 private:
  std::string value_;
  static void Wipe(std::string& s);
};

class CedarClient {
 public:
  explicit CedarClient(ClientConfig cfg);

  // Advanced primary API surface.
  ApiResponse bootstrap_channel();
  ApiResponse authenticate_identity(const std::string& username,
                                    const SecureString& password);
  ApiResponse enroll_identity(const std::string& username,
                              const SecureString& password,
                              const std::string& email,
                              const std::string& license_key);
  ApiResponse activate_entitlement(const std::string& license_key);
  ApiResponse attest_session();
  ApiResponse dispatch_telemetry(const std::string& title, const std::string& message);
  ApiResponse fetch_module(const std::string& remote_path_or_url,
                           const std::string& output_file);

  // Compatibility aliases.
  inline ApiResponse init() { return bootstrap_channel(); }
  inline ApiResponse login(const std::string& username, const SecureString& password) {
    return authenticate_identity(username, password);
  }
  inline ApiResponse regstr(const std::string& username,
                            const SecureString& password,
                            const std::string& email,
                            const std::string& license_key) {
    return enroll_identity(username, password, email, license_key);
  }
  inline ApiResponse license(const std::string& license_key) {
    return activate_entitlement(license_key);
  }
  inline ApiResponse check() { return attest_session(); }
  inline ApiResponse log(const std::string& title, const std::string& message) {
    return dispatch_telemetry(title, message);
  }
  inline ApiResponse download(const std::string& remote_path_or_url,
                              const std::string& output_file) {
    return fetch_module(remote_path_or_url, output_file);
  }

  const SessionState& session() const;
  std::string hwid() const;

  bool debugger_detected() const;
  bool vm_detected() const;
  bool local_security_gate(std::string* reason) const;
  bool process_blacklist_hit(const std::vector<std::string>& blocked_names,
                             std::string* matched_name) const;
  bool memory_integrity_ok(const std::string& expected_sha256_hex,
                           std::string* actual_sha256_hex) const;

  static std::string encrypt_string(const std::string& plaintext,
                                    const SecureString& passphrase);
  static std::string decrypt_string(const std::string& cipher_blob_b64,
                                    const SecureString& passphrase);

  static std::string ObfuscateXor(const std::string& in, unsigned char key);
  static std::string DeobfuscateXor(const std::string& in, unsigned char key);

 private:
  struct HttpResult {
    bool network_ok = false;
    long status = 0;
    std::string body;
  };

  ClientConfig cfg_;
  SessionState session_;

  static std::string MakeNonce();
  static std::int64_t Now();
  static std::string JsonEscape(const std::string& in);
  static std::string JsonGetString(const std::string& json, const std::string& key);
  static bool JsonGetBool(const std::string& json, const std::string& key, bool default_value);
  static std::int64_t JsonGetInt(const std::string& json,
                                 const std::string& key,
                                 std::int64_t default_value);
  static std::map<std::string, std::string> ParseInterestingFields(const std::string& json);
  static std::string NormalizeError(const std::string& payload);

  std::string SignRequest(const std::string& body,
                          const std::string& nonce,
                          std::int64_t timestamp) const;

  HttpResult HttpJson(const std::string& method,
                      const std::string& path,
                      const std::string& body,
                      const std::vector<std::string>& extra_headers) const;

  HttpResult HttpDownload(const std::string& full_or_relative_url,
                          const std::string& output_file,
                          const std::vector<std::string>& extra_headers) const;

  ApiResponse ValidateEntitlement() const;
};

}  // namespace cedar
